/*
 * Copyright (c) 2004-2006 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_sa.c 7794 2006-06-07 15:09:21Z halr $
 */


/*
 * Abstract:
 *    Implementation of osm_sa_t.
 * This object represents the Subnet Administrator object.
 * This object is part of the opensm family of objects.
 *
 * Environment:
 *    Linux User Mode
 *
 * $Revision: 1.14 $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <complib/cl_qmap.h>
#include <complib/cl_passivelock.h>
#include <complib/cl_debug.h>
#include <iba/ib_types.h>
#include <opensm/osm_sa.h>
#include <opensm/osm_sa_response.h>
#include <opensm/osm_madw.h>
#include <opensm/osm_log.h>
#include <opensm/osm_subnet.h>
#include <opensm/osm_mad_pool.h>
#include <opensm/osm_msgdef.h>

#define  OSM_SA_INITIAL_TID_VALUE 0xabc

/**********************************************************************
 **********************************************************************/
void
osm_sa_construct(
  IN osm_sa_t* const p_sa )
{
  memset( p_sa, 0, sizeof(*p_sa) );
  p_sa->state = OSM_SA_STATE_INIT;
  p_sa->sa_trans_id = OSM_SA_INITIAL_TID_VALUE;

  osm_sa_mad_ctrl_construct( &p_sa->mad_ctrl );
  osm_sa_resp_construct( &p_sa->resp );

  osm_nr_rcv_construct( &p_sa->nr_rcv);
  osm_nr_rcv_ctrl_construct( &p_sa->nr_rcv_ctrl );

  osm_pir_rcv_construct( &p_sa->pir_rcv );
  osm_pir_rcv_ctrl_construct( &p_sa->pir_rcv_ctrl );

  osm_gir_rcv_construct( &p_sa->gir_rcv );
  osm_gir_rcv_ctrl_construct( &p_sa->gir_rcv_ctrl );

  osm_lr_rcv_construct( &p_sa->lr_rcv );
  osm_lr_rcv_ctrl_construct( &p_sa->lr_rcv_ctrl );

  osm_pr_rcv_construct( &p_sa->pr_rcv );
  osm_pr_rcv_ctrl_construct( &p_sa->pr_rcv_ctrl );

#if defined (VENDOR_RMPP_SUPPORT) && defined (DUAL_SIDED_RMPP)
  osm_mpr_rcv_construct( &p_sa->mpr_rcv );
  osm_mpr_rcv_ctrl_construct( &p_sa->mpr_rcv_ctrl );
#endif

  osm_smir_rcv_construct( &p_sa->smir_rcv );
  osm_smir_ctrl_construct( &p_sa->smir_ctrl );

  osm_mcmr_rcv_construct(&p_sa->mcmr_rcv );
  osm_mcmr_rcv_ctrl_construct(&p_sa->mcmr_rcv_ctlr);

  osm_sr_rcv_construct( &p_sa->sr_rcv );
  osm_sr_rcv_ctrl_construct( &p_sa->sr_rcv_ctrl );

  osm_infr_rcv_construct( &p_sa->infr_rcv );
  osm_infr_rcv_ctrl_construct( &p_sa->infr_rcv_ctrl );

  osm_vlarb_rec_rcv_construct( &p_sa->vlarb_rec_rcv );
  osm_vlarb_rec_rcv_ctrl_construct( &p_sa->vlarb_rec_rcv_ctrl );

  osm_slvl_rec_rcv_construct( &p_sa->slvl_rec_rcv );
  osm_slvl_rec_rcv_ctrl_construct( &p_sa->slvl_rec_rcv_ctrl );

  osm_pkey_rec_rcv_construct( &p_sa->pkey_rec_rcv );
  osm_pkey_rec_rcv_ctrl_construct( &p_sa->pkey_rec_rcv_ctrl );

  osm_lftr_rcv_construct( &p_sa->lftr_rcv );
  osm_lftr_rcv_ctrl_construct( &p_sa->lftr_rcv_ctrl );
}

/**********************************************************************
 **********************************************************************/
void
osm_sa_shutdown(
  IN osm_sa_t* const p_sa )
{
  ib_api_status_t status;
  OSM_LOG_ENTER( p_sa->p_log, osm_sa_shutdown );

  /* unbind from the mad service */
  status = osm_sa_mad_ctrl_unbind( &p_sa->mad_ctrl );

  /* remove any registered dispatcher message */
  osm_nr_rcv_ctrl_destroy( &p_sa->nr_rcv_ctrl );
  osm_pir_rcv_ctrl_destroy( &p_sa->pir_rcv_ctrl );
  osm_gir_rcv_ctrl_destroy( &p_sa->gir_rcv_ctrl );
  osm_lr_rcv_ctrl_destroy( &p_sa->lr_rcv_ctrl );
  osm_pr_rcv_ctrl_destroy( &p_sa->pr_rcv_ctrl );
#if defined (VENDOR_RMPP_SUPPORT) && defined (DUAL_SIDED_RMPP)
  osm_mpr_rcv_ctrl_destroy( &p_sa->mpr_rcv_ctrl );
#endif
  osm_smir_ctrl_destroy( &p_sa->smir_ctrl );
  osm_mcmr_rcv_ctrl_destroy( &p_sa->mcmr_rcv_ctlr);
  osm_sr_rcv_ctrl_destroy( &p_sa->sr_rcv_ctrl );
  osm_infr_rcv_ctrl_destroy( &p_sa->infr_rcv_ctrl );
  osm_vlarb_rec_rcv_ctrl_destroy( &p_sa->vlarb_rec_rcv_ctrl );
  osm_slvl_rec_rcv_ctrl_destroy( &p_sa->slvl_rec_rcv_ctrl );
  osm_pkey_rec_rcv_ctrl_destroy( &p_sa->pkey_rec_rcv_ctrl );
  osm_lftr_rcv_ctrl_destroy( &p_sa->lftr_rcv_ctrl );
  osm_sa_mad_ctrl_destroy( &p_sa->mad_ctrl );

  OSM_LOG_EXIT( p_sa->p_log );
}

/**********************************************************************
 **********************************************************************/
void
osm_sa_destroy(
  IN osm_sa_t* const p_sa )
{
  OSM_LOG_ENTER( p_sa->p_log, osm_sa_destroy );

  p_sa->state = OSM_SA_STATE_INIT;

  osm_nr_rcv_destroy( &p_sa->nr_rcv );
  osm_pir_rcv_destroy( &p_sa->pir_rcv );
  osm_gir_rcv_destroy( &p_sa->gir_rcv );
  osm_lr_rcv_destroy( &p_sa->lr_rcv );
  osm_pr_rcv_destroy( &p_sa->pr_rcv );
#if defined (VENDOR_RMPP_SUPPORT) && defined (DUAL_SIDED_RMPP)
  osm_mpr_rcv_destroy( &p_sa->mpr_rcv );
#endif
  osm_smir_rcv_destroy( &p_sa->smir_rcv );
  osm_mcmr_rcv_destroy(&p_sa->mcmr_rcv);
  osm_sr_rcv_destroy( &p_sa->sr_rcv );
  osm_infr_rcv_destroy( &p_sa->infr_rcv );
  osm_vlarb_rec_rcv_destroy( &p_sa->vlarb_rec_rcv );
  osm_slvl_rec_rcv_destroy( &p_sa->slvl_rec_rcv );
  osm_pkey_rec_rcv_destroy( &p_sa->pkey_rec_rcv );
  osm_lftr_rcv_destroy( &p_sa->lftr_rcv );
  osm_sa_resp_destroy( &p_sa->resp );

  OSM_LOG_EXIT( p_sa->p_log );
}

/**********************************************************************
 **********************************************************************/
ib_api_status_t
osm_sa_init(
  IN osm_sm_t* const p_sm,
  IN osm_sa_t* const p_sa,
  IN osm_subn_t* const p_subn,
  IN osm_vendor_t* const p_vendor,
  IN osm_mad_pool_t* const p_mad_pool,
  IN osm_log_t* const p_log,
  IN osm_stats_t* const p_stats,
  IN cl_dispatcher_t* const p_disp,
  IN cl_plock_t* const p_lock )
{
  ib_api_status_t status = IB_SUCCESS;

  OSM_LOG_ENTER( p_log, osm_sa_init );

  p_sa->p_subn = p_subn;
  p_sa->p_vendor = p_vendor;
  p_sa->p_mad_pool = p_mad_pool;
  p_sa->p_log = p_log;
  p_sa->p_disp = p_disp;
  p_sa->p_lock = p_lock;

  p_sa->state = OSM_SA_STATE_READY;

  status = osm_sa_resp_init(&p_sa->resp,
                            p_sa->p_mad_pool,
                            p_log);
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_sa_mad_ctrl_init(
    &p_sa->mad_ctrl,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_sa->p_vendor,
    p_subn,
    p_log,
    p_stats,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_cpi_rcv_init(
    &p_sa->cpi_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_cpi_rcv_ctrl_init(
    &p_sa->cpi_rcv_ctrl,
    &p_sa->cpi_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_nr_rcv_init(
    &p_sa->nr_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_nr_rcv_ctrl_init(
    &p_sa->nr_rcv_ctrl,
    &p_sa->nr_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_pir_rcv_init(
    &p_sa->pir_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_pir_rcv_ctrl_init(
    &p_sa->pir_rcv_ctrl,
    &p_sa->pir_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_gir_rcv_init(
    &p_sa->gir_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_gir_rcv_ctrl_init(
    &p_sa->gir_rcv_ctrl,
    &p_sa->gir_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_lr_rcv_init(
    &p_sa->lr_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_lr_rcv_ctrl_init(
    &p_sa->lr_rcv_ctrl,
    &p_sa->lr_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_pr_rcv_init(
    &p_sa->pr_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_pr_rcv_ctrl_init(
    &p_sa->pr_rcv_ctrl,
    &p_sa->pr_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

#if defined (VENDOR_RMPP_SUPPORT) && defined (DUAL_SIDED_RMPP)
  status = osm_mpr_rcv_init(
    &p_sa->mpr_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool, 
    p_subn,
    p_log,
    p_lock );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_mpr_rcv_ctrl_init(
    &p_sa->mpr_rcv_ctrl,
    &p_sa->mpr_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;
#endif

  status = osm_smir_rcv_init(
    &p_sa->smir_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_stats,
    p_log,
    p_lock );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_smir_ctrl_init(
    &p_sa->smir_ctrl,
    &p_sa->smir_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_mcmr_rcv_init(
    p_sm,
    &p_sa->mcmr_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock);
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_mcmr_rcv_ctrl_init(
    &p_sa->mcmr_rcv_ctlr,
    &p_sa->mcmr_rcv,
    p_log,
    p_disp);
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_sr_rcv_init(
    &p_sa->sr_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock);
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_sr_rcv_ctrl_init(
    &p_sa->sr_rcv_ctrl,
    &p_sa->sr_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_infr_rcv_init(
    &p_sa->infr_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock);
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_infr_rcv_ctrl_init(
    &p_sa->infr_rcv_ctrl,
    &p_sa->infr_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_vlarb_rec_rcv_init(
    &p_sa->vlarb_rec_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock);
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_vlarb_rec_rcv_ctrl_init(
    &p_sa->vlarb_rec_rcv_ctrl,
    &p_sa->vlarb_rec_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_slvl_rec_rcv_init(
    &p_sa->slvl_rec_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock);
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_slvl_rec_rcv_ctrl_init(
    &p_sa->slvl_rec_rcv_ctrl,
    &p_sa->slvl_rec_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_pkey_rec_rcv_init(
    &p_sa->pkey_rec_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock);
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_pkey_rec_rcv_ctrl_init(
    &p_sa->pkey_rec_rcv_ctrl,
    &p_sa->pkey_rec_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_lftr_rcv_init(
    &p_sa->lftr_rcv,
    &p_sa->resp,
    p_sa->p_mad_pool,
    p_subn,
    p_log,
    p_lock);
  if( status != IB_SUCCESS )
    goto Exit;

  status = osm_lftr_rcv_ctrl_init(
    &p_sa->lftr_rcv_ctrl,
    &p_sa->lftr_rcv,
    p_log,
    p_disp );
  if( status != IB_SUCCESS )
    goto Exit;

 Exit:
  OSM_LOG_EXIT( p_log );
  return( status );
}

/**********************************************************************
 **********************************************************************/
ib_api_status_t
osm_sa_bind(
  IN osm_sa_t* const p_sa,
  IN const ib_net64_t port_guid )
{
  ib_api_status_t status;

  OSM_LOG_ENTER( p_sa->p_log, osm_sa_bind );

  status = osm_sa_mad_ctrl_bind(
    &p_sa->mad_ctrl, port_guid );

  OSM_LOG_EXIT( p_sa->p_log );
  return( status );
}
